-- Check if all required tables and columns exist before creating the view
DO $$
DECLARE
    tables_exist BOOLEAN := TRUE;
    columns_exist BOOLEAN := TRUE;
BEGIN
    -- Check if all required tables exist
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'mcx_conference') THEN
        RAISE NOTICE 'Table mcx_conference does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'mcx_participant') THEN
        RAISE NOTICE 'Table mcx_participant does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'sip_registrations') THEN
        RAISE NOTICE 'Table sip_registrations does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'location') THEN
        RAISE NOTICE 'Table location does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'channels') THEN
        RAISE NOTICE 'Table channels does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_name = 'basic_calls') THEN
        RAISE NOTICE 'Table basic_calls does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    IF NOT EXISTS (SELECT 1 FROM information_schema.tables WHERE table_schema = 'pbx' AND table_name = 'extensions') THEN
        RAISE NOTICE 'Table pbx.extensions does not exist. Skipping view creation.';
        tables_exist := FALSE;
    END IF;
    
    -- Only check columns if all tables exist
    IF tables_exist THEN
        -- Check if all required columns exist in mcx_conference
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_conference' AND column_name = 'conference_no') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_conference' AND column_name = 'enable_simulation') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_conference' AND column_name = 'control_with') THEN
            RAISE NOTICE 'Required columns missing in mcx_conference table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Check if all required columns exist in mcx_participant
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_participant' AND column_name = 'participant_no') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_participant' AND column_name = 'connected_ip_address') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_participant' AND column_name = 'keep_alive_epoch') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'mcx_participant' AND column_name = 'join_conference') THEN
            RAISE NOTICE 'Required columns missing in mcx_participant table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Check if all required columns exist in sip_registrations
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'sip_registrations' AND column_name = 'sip_user') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'sip_registrations' AND column_name = 'status') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'sip_registrations' AND column_name = 'expires') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'sip_registrations' AND column_name = 'user_agent') THEN
            RAISE NOTICE 'Required columns missing in sip_registrations table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Check if all required columns exist in location
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'location' AND column_name = 'username') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'location' AND column_name = 'expires') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'location' AND column_name = 'user_agent') THEN
            RAISE NOTICE 'Required columns missing in location table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Check if all required columns exist in channels
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'channels' AND column_name = 'initial_dest') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'channels' AND column_name = 'initial_cid_num') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'channels' AND column_name = 'application') THEN
            RAISE NOTICE 'Required columns missing in channels table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Check if all required columns exist in basic_calls
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'basic_calls' AND column_name = 'uuid') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'basic_calls' AND column_name = 'coral_original_num') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_name = 'basic_calls' AND column_name = 'b_coral_original_num') THEN
            RAISE NOTICE 'Required columns missing in basic_calls table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Check if all required columns exist in pbx.extensions
        IF NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'extension') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'designation') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'device_id') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'battery_percent') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'rsrp_sim1') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'rsrp_sim2') OR
           NOT EXISTS (SELECT 1 FROM information_schema.columns WHERE table_schema = 'pbx' AND table_name = 'extensions' AND column_name = 'device_lastupdated') THEN
            RAISE NOTICE 'Required columns missing in pbx.extensions table. Skipping view creation.';
            columns_exist := FALSE;
        END IF;
        
        -- Create the view only if all prerequisites are met
        IF tables_exist AND columns_exist THEN
            RAISE NOTICE 'All required tables and columns exist. Creating view mcx_conference_status.';
            
            -- Drop existing view if it exists
            DROP VIEW IF EXISTS mcx_conference_status;
            
            -- Create the view
            CREATE VIEW mcx_conference_status AS
            SELECT *
            FROM (
                SELECT 
                    a.conference_no,
                    CASE 
                        WHEN a.enable_simulation = TRUE THEN 'Yes' 
                        ELSE 'No' 
                    END AS SimulationOn,

                    b.participant_no,
                    b.connected_ip_address,

                    CASE 
                        WHEN (
                            COALESCE(c.sip_user, l.username, '') <> ''
                            OR TO_TIMESTAMP(b.keep_alive_epoch) >= (NOW() AT TIME ZONE 'UTC') - INTERVAL '15 minutes'
                            OR TO_TIMESTAMP(f.device_lastupdated) >= (NOW() AT TIME ZONE 'UTC') - INTERVAL '15 minutes'
                        )
                        THEN 'Registered'
                        ELSE 'UnRegistered'
                    END AS RegisteredStatus,

                    COALESCE(c.status) AS transport,

                    -- Convert BIGINT (epoch time) to TIMESTAMP and coalesce with l.expires
                    COALESCE(TO_TIMESTAMP(c.expires), l.expires) AS RegistrationExpiredOn,

                    CASE 
                        WHEN (
                            COALESCE(d.initial_cid_num, '') <> ''
                            OR TO_TIMESTAMP(b.keep_alive_epoch) >= (NOW() AT TIME ZONE 'UTC') - INTERVAL '15 minutes'
                            OR TO_TIMESTAMP(f.device_lastupdated) >= (NOW() AT TIME ZONE 'UTC') - INTERVAL '15 minutes'
                        )
                        THEN 'In Conference'
                        WHEN (
                            COALESCE(e.uuid, '') <> '' 
                            AND COALESCE(d.initial_cid_num, '') = ''
                        )
                        THEN 'In Private Call'
                        ELSE 'Not In Conference'
                    END AS ConferenceStatus,

                    COALESCE(c.user_agent, l.user_agent) AS user_agent,
                    f.designation,
                    f.device_id,
                    f.battery_percent,
                    f.rsrp_sim1,
                    f.rsrp_sim2,
                    f.device_lastupdated,
                    a.control_with,
                    a.video_stream_id,
                    a.video_stream_status,
                    a.video_stream_by

                FROM mcx_conference a
                INNER JOIN mcx_participant b 
                    ON a.conference_no = b.join_conference
                LEFT JOIN sip_registrations c 
                    ON b.participant_no = c.sip_user
                LEFT JOIN location l 
                    ON b.participant_no = l.username
                LEFT JOIN channels d 
                    ON b.participant_no = d.initial_dest 
                    AND b.join_conference = d.initial_cid_num 
                    AND d.application = 'conference'
                LEFT JOIN basic_calls e 
                    ON b.participant_no = e.coral_original_num 
                    OR b.participant_no = e.b_coral_original_num
                LEFT JOIN pbx.extensions f 
                    ON b.participant_no = f.extension
            ) a
            ORDER BY 
                a.conference_no, 
                a.user_agent, 
                a.ConferenceStatus, 
                a.RegisteredStatus;
            
            RAISE NOTICE 'View mcx_conference_status created successfully.';
        ELSE
            RAISE NOTICE 'Prerequisites not met. View mcx_conference_status was not created.';
        END IF;
    ELSE
        RAISE NOTICE 'Required tables missing. View mcx_conference_status was not created.';
    END IF;
END;
$$;