/*
 * Decompiled with CFR 0.152.
 */
package com.coraltele.telemetry.service.retention;

import com.coraltele.telemetry.helper.Constants;
import com.coraltele.telemetry.model.retention.DatabaseConfig;
import com.coraltele.telemetry.model.retention.DbRetentionConfig;
import com.coraltele.telemetry.model.retention.TableConfig;
import com.coraltele.telemetry.model.retention.TruncateConfig;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.sql.Timestamp;
import java.time.Duration;
import java.time.Instant;
import java.util.List;
import javax.sql.DataSource;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.core.io.ClassPathResource;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.stereotype.Service;
import org.springframework.util.StringUtils;

@Service
public class RetentionService {
    private static final Logger logger = LogManager.getLogger(RetentionService.class);
    private DbRetentionConfig retentionConfig;

    public void initialize() {
        try {
            this.loadRetentionConfig();
            logger.info("DB Retention Service initialized successfully");
        }
        catch (Exception e) {
            logger.error("Failed to initialize DB Retention Service", (Throwable)e);
        }
    }

    private void loadRetentionConfig() throws IOException {
        ClassPathResource resource = new ClassPathResource("default-data/db-retention.json");
        try (InputStream inputStream = resource.getInputStream();){
            ObjectMapper mapper = new ObjectMapper();
            this.retentionConfig = (DbRetentionConfig)mapper.readValue(inputStream, DbRetentionConfig.class);
            logger.info("Loaded retention configuration successfully");
        }
    }

    public void executeRetentionCleanup() {
        if (this.retentionConfig == null) {
            logger.warn("Retention configuration not loaded. Initializing...");
            this.initialize();
            if (this.retentionConfig == null) {
                logger.error("Unable to load retention configuration. Skipping cleanup.");
                return;
            }
        }
        logger.info("Starting DB retention cleanup process");
        Instant startTime = Instant.now();
        try {
            this.processRawCdrRetention();
            this.processProcessedCdrRetention();
            this.processTruncateTables();
            long duration = Duration.between(startTime, Instant.now()).toMillis();
            logger.info("DB retention cleanup completed in {} ms", (Object)duration);
        }
        catch (Exception e) {
            logger.error("Error during retention cleanup process", (Throwable)e);
        }
    }

    private void processRawCdrRetention() {
        int retentionDays = Constants.RAW_CDR_DATA_DELETION_TIME_PERIOD;
        logger.info("Processing Raw CDR retention with {} days retention period", (Object)retentionDays);
        if (this.retentionConfig.getRawCdr() != null && this.retentionConfig.getRawCdr().getDatabases() != null) {
            for (DatabaseConfig dbConfig : this.retentionConfig.getRawCdr().getDatabases()) {
                this.processDatabase(dbConfig, retentionDays, "Raw CDR");
            }
        }
    }

    private void processProcessedCdrRetention() {
        int retentionDays = Constants.PROCESSED_CDR_DATA_DELETION_TIME_PERIOD;
        logger.info("Processing Processed CDR retention with {} days retention period", (Object)retentionDays);
        if (this.retentionConfig.getProcessedCdr() != null && this.retentionConfig.getProcessedCdr().getDatabases() != null) {
            for (DatabaseConfig dbConfig : this.retentionConfig.getProcessedCdr().getDatabases()) {
                this.processDatabase(dbConfig, retentionDays, "Processed CDR");
            }
        }
    }

    private void processDatabase(DatabaseConfig dbConfig, int retentionDays, String category) {
        String databaseName = dbConfig.getName();
        logger.info("Processing {} retention for database: {}", (Object)category, (Object)databaseName);
        JdbcTemplate jdbcTemplate = this.getJdbcTemplateForDatabase(databaseName);
        if (jdbcTemplate == null) {
            logger.warn("Unable to get JDBC template for database: {}. Skipping.", (Object)databaseName);
            return;
        }
        if (dbConfig.getTables() != null) {
            for (TableConfig tableConfig : dbConfig.getTables()) {
                this.deleteOldRecords(jdbcTemplate, databaseName, tableConfig, retentionDays, category);
            }
        }
    }

    private void deleteOldRecords(JdbcTemplate jdbcTemplate, String databaseName, TableConfig tableConfig, int retentionDays, String category) {
        String tableName = tableConfig.getName();
        String epochColumn = tableConfig.getEpochColumn();
        String fileNameColumn = tableConfig.getFileName();
        Instant cutoffInstant = Instant.now().minusSeconds(((long)retentionDays + 1L) * 24L * 60L * 60L);
        String epochType = tableConfig.getEpochType() == null ? "seconds" : tableConfig.getEpochType();
        try {
            int deletedRows;
            if (Boolean.TRUE.equals(Constants.DELETE_FILES_WITH_RECORDS) && StringUtils.hasText((String)fileNameColumn)) {
                this.deleteAssociatedFiles(jdbcTemplate, tableName, epochColumn, fileNameColumn, cutoffInstant, epochType, databaseName, category);
            }
            String deleteSql = String.format("DELETE FROM %s WHERE %s < ?", tableName, epochColumn);
            if ("timestamp".equalsIgnoreCase(epochType)) {
                deletedRows = jdbcTemplate.update(deleteSql, new Object[]{Timestamp.from(cutoffInstant)});
            } else {
                long cutoffEpochSeconds = cutoffInstant.getEpochSecond();
                deletedRows = jdbcTemplate.update(deleteSql, new Object[]{cutoffEpochSeconds});
            }
            if (deletedRows > 0) {
                logger.info("{} - Database: {}, Table: {}, Deleted {} records older than {} days", (Object)category, (Object)databaseName, (Object)tableName, (Object)deletedRows, (Object)retentionDays);
            } else {
                logger.debug("{} - Database: {}, Table: {}, No records to delete", (Object)category, (Object)databaseName, (Object)tableName);
            }
        }
        catch (Exception e) {
            logger.error("{} - Failed to delete records from {}.{}: {}", (Object)category, (Object)databaseName, (Object)tableName, (Object)e.getMessage(), (Object)e);
        }
    }

    private void deleteAssociatedFiles(JdbcTemplate jdbcTemplate, String tableName, String epochColumn, String fileNameColumn, Instant cutoffInstant, String epochType, String databaseName, String category) {
        String selectSql = String.format("SELECT %s FROM %s WHERE %s < ? AND %s IS NOT NULL AND %s != ''", fileNameColumn, tableName, epochColumn, fileNameColumn, fileNameColumn);
        try {
            List filePaths;
            if ("timestamp".equalsIgnoreCase(epochType)) {
                filePaths = jdbcTemplate.queryForList(selectSql, String.class, new Object[]{Timestamp.from(cutoffInstant)});
            } else {
                long cutoffEpochSeconds = cutoffInstant.getEpochSecond();
                filePaths = jdbcTemplate.queryForList(selectSql, String.class, new Object[]{cutoffEpochSeconds});
            }
            if (filePaths.isEmpty()) {
                logger.debug("{} - Database: {}, Table: {}, No associated files to delete", (Object)category, (Object)databaseName, (Object)tableName);
                return;
            }
            int deletedFiles = 0;
            int skippedFiles = 0;
            for (String filePath : filePaths) {
                if (!StringUtils.hasText((String)filePath)) continue;
                File file = new File(filePath);
                if (file.exists()) {
                    if (file.delete()) {
                        ++deletedFiles;
                        continue;
                    }
                    logger.warn("{} - Failed to delete file: {}", (Object)category, (Object)filePath);
                    continue;
                }
                ++skippedFiles;
            }
            logger.info("{} - Database: {}, Table: {}, Deleted {} files, Skipped {} non-existent files", (Object)category, (Object)databaseName, (Object)tableName, (Object)deletedFiles, (Object)skippedFiles);
        }
        catch (Exception e) {
            logger.error("{} - Failed to delete associated files for {}.{}: {}", (Object)category, (Object)databaseName, (Object)tableName, (Object)e.getMessage(), (Object)e);
        }
    }

    private void processTruncateTables() {
        if (this.retentionConfig.getTruncateTables() == null) {
            return;
        }
        logger.info("Processing table truncation");
        for (TruncateConfig truncateConfig : this.retentionConfig.getTruncateTables()) {
            String databaseName = truncateConfig.getDatabase();
            JdbcTemplate jdbcTemplate = this.getJdbcTemplateForDatabase(databaseName);
            if (jdbcTemplate == null) {
                logger.warn("Unable to get JDBC template for database: {}. Skipping truncation.", (Object)databaseName);
                continue;
            }
            if (truncateConfig.getTables() == null) continue;
            for (String tableName : truncateConfig.getTables()) {
                this.truncateTable(jdbcTemplate, databaseName, tableName);
            }
        }
    }

    private void truncateTable(JdbcTemplate jdbcTemplate, String databaseName, String tableName) {
        String truncateSql = String.format("TRUNCATE TABLE %s", tableName);
        try {
            jdbcTemplate.execute(truncateSql);
            logger.info("Truncated table: {}.{}", (Object)databaseName, (Object)tableName);
        }
        catch (Exception e) {
            logger.error("Failed to truncate table {}.{}: {}", (Object)databaseName, (Object)tableName, (Object)e.getMessage(), (Object)e);
        }
    }

    private JdbcTemplate getJdbcTemplateForDatabase(String databaseName) {
        try {
            String url = String.format("jdbc:postgresql://127.0.0.1:5432/%s?ApplicationName=telemetry-db-retention", databaseName);
            DriverManagerDataSource dataSource = new DriverManagerDataSource();
            dataSource.setDriverClassName("org.postgresql.Driver");
            dataSource.setUrl(url);
            dataSource.setUsername("postgres");
            dataSource.setPassword("");
            return new JdbcTemplate((DataSource)dataSource);
        }
        catch (Exception e) {
            logger.error("Failed to create JdbcTemplate for database: {}", (Object)databaseName, (Object)e);
            return null;
        }
    }

    public void reloadConfiguration() {
        logger.info("Reloading retention configuration");
        try {
            this.loadRetentionConfig();
            logger.info("Retention configuration reloaded successfully");
        }
        catch (IOException e) {
            logger.error("Failed to reload retention configuration", (Throwable)e);
        }
    }
}

