/*
 * Decompiled with CFR 0.152.
 */
package com.coraltele.telemetry.scheduler;

import com.coraltele.telemetry.helper.Constants;
import com.coraltele.telemetry.helper.HttpUtils;
import com.coraltele.telemetry.service.retention.RetentionService;
import java.io.File;
import java.time.LocalTime;
import java.time.format.DateTimeParseException;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledFuture;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.TaskScheduler;
import org.springframework.scheduling.Trigger;
import org.springframework.scheduling.support.CronTrigger;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

@Component
public class CronScheduler {
    private static final Logger logger = LogManager.getLogger(CronScheduler.class);
    private static final String BACKUP_DIRECTORY = "/var/backups/coral";
    private static final Pattern BACKUP_FULL_PATTERN = Pattern.compile("backup_full_(\\d{8}_\\d{6})\\.tar\\.gz$");
    private static final Pattern BACKUP_CONFIG_PATTERN = Pattern.compile("backup_config_(\\d{8}_\\d{6})\\.tar\\.gz$");
    @Autowired
    private TaskScheduler taskScheduler;
    @Autowired
    private RetentionService retentionService;
    private final Map<String, ScheduledFuture<?>> scheduledTasks = new ConcurrentHashMap();

    @PostConstruct
    public void initializeScheduledTasks() {
        logger.info("Initializing CronScheduler with default tasks");
        try {
            this.retentionService.initialize();
        }
        catch (Exception e) {
            logger.error("Failed to initialize DbRetentionService", (Throwable)e);
        }
        String cron = this.buildCronExpression(Constants.DB_RETENTION_CLEANUP_TIME, "0 0 2 * * ?");
        this.scheduleTask("dbRetentionCleanup", cron, () -> this.executeDbRetentionCleanup());
        if (StringUtils.hasText((String)Constants.AUTO_BACKUP_FULL)) {
            String backupFullCron = this.buildCronExpression(Constants.AUTO_BACKUP_FULL, "0 0 3 * * ?");
            this.scheduleTask("autoBackupFull", backupFullCron, () -> this.triggerAutoBackupFull());
        }
        if (StringUtils.hasText((String)Constants.AUTO_BACKUP_CONFIGURATION)) {
            String backupConfigCron = this.buildCronExpression(Constants.AUTO_BACKUP_CONFIGURATION, "0 0 3 * * ?");
            this.scheduleTask("autoBackupConfiguration", backupConfigCron, () -> this.triggerAutoBackupConfiguration());
        }
        logger.info("CronScheduler initialized with {} scheduled tasks", (Object)this.scheduledTasks.size());
    }

    public boolean scheduleTask(String taskId, String cronExpression, Runnable task) {
        try {
            this.cancelTask(taskId);
            ScheduledFuture scheduledTask = this.taskScheduler.schedule(task, (Trigger)new CronTrigger(cronExpression));
            this.scheduledTasks.put(taskId, scheduledTask);
            logger.info("Scheduled task '{}' with cron expression: {}", (Object)taskId, (Object)cronExpression);
            return true;
        }
        catch (Exception e) {
            logger.error("Failed to schedule task '{}' with cron: {}", (Object)taskId, (Object)cronExpression, (Object)e);
            return false;
        }
    }

    public boolean cancelTask(String taskId) {
        ScheduledFuture scheduledTask = (ScheduledFuture)this.scheduledTasks.get(taskId);
        if (scheduledTask != null) {
            scheduledTask.cancel(false);
            this.scheduledTasks.remove(taskId);
            logger.info("Cancelled scheduled task: {}", (Object)taskId);
            return true;
        }
        return false;
    }

    public boolean rescheduleTask(String taskId, String newCronExpression, Runnable task) {
        logger.info("Rescheduling task '{}' with new cron: {}", (Object)taskId, (Object)newCronExpression);
        this.cancelTask(taskId);
        return this.scheduleTask(taskId, newCronExpression, task);
    }

    public String[] getScheduledTaskIds() {
        return this.scheduledTasks.keySet().toArray(new String[0]);
    }

    public boolean isTaskScheduled(String taskId) {
        ScheduledFuture task = (ScheduledFuture)this.scheduledTasks.get(taskId);
        return task != null && !task.isCancelled() && !task.isDone();
    }

    private void executeDbRetentionCleanup() {
        logger.info("DB Retention Scheduler triggered");
        try {
            this.retentionService.executeRetentionCleanup();
            logger.info("DB Retention Scheduler completed successfully");
        }
        catch (Exception e) {
            logger.error("Error in DB Retention Scheduler", (Throwable)e);
        }
    }

    public void triggerDbRetentionCleanup() {
        logger.info("Manually triggering DB Retention cleanup");
        this.executeDbRetentionCleanup();
    }

    public void triggerAutoBackupFull() {
        logger.info("Manually triggering Auto Backup Full");
        boolean success = HttpUtils.executeAutoBackupFull();
        if (success) {
            logger.info("Auto Backup Full completed successfully, cleaning up old backups");
            this.cleanupOldBackups(BACKUP_FULL_PATTERN, Constants.AUTO_BACKUP_FULL_COPIES.intValue());
        } else {
            logger.error("Auto Backup Full failed, skipping cleanup");
        }
    }

    public void triggerAutoBackupConfiguration() {
        logger.info("Manually triggering Auto Backup Configuration");
        boolean success = HttpUtils.executeAutoBackupConfiguration();
        if (success) {
            logger.info("Auto Backup Configuration completed successfully, cleaning up old backups");
            this.cleanupOldBackups(BACKUP_CONFIG_PATTERN, Constants.AUTO_BACKUP_CONFIGURATION_COPIES.intValue());
        } else {
            logger.error("Auto Backup Configuration failed, skipping cleanup");
        }
    }

    private void cleanupOldBackups(Pattern pattern, int copiesToKeep) {
        if (copiesToKeep <= 0) {
            logger.warn("Invalid copiesToKeep value: {}, skipping cleanup", (Object)copiesToKeep);
            return;
        }
        File backupDir = new File(BACKUP_DIRECTORY);
        if (!backupDir.exists() || !backupDir.isDirectory()) {
            logger.warn("Backup directory does not exist: {}", (Object)BACKUP_DIRECTORY);
            return;
        }
        File[] files = backupDir.listFiles();
        if (files == null || files.length == 0) {
            logger.info("No backup files found in {}", (Object)BACKUP_DIRECTORY);
            return;
        }
        List backupFiles = Arrays.stream(files).filter(f -> f.isFile() && pattern.matcher(f.getName()).matches()).sorted(Comparator.comparing(arg_0 -> this.extractTimestamp(arg_0)).reversed()).collect(Collectors.toList());
        logger.info("Found {} backup files matching pattern, keeping {} copies", (Object)backupFiles.size(), (Object)copiesToKeep);
        if (backupFiles.size() > copiesToKeep) {
            List filesToDelete = backupFiles.subList(copiesToKeep, backupFiles.size());
            for (File backupFile : filesToDelete) {
                this.deleteBackupPair(backupFile);
            }
            logger.info("Cleanup completed, deleted {} old backup pairs", (Object)filesToDelete.size());
        } else {
            logger.info("No old backups to delete, current count ({}) is within limit ({})", (Object)backupFiles.size(), (Object)copiesToKeep);
        }
    }

    private void deleteBackupPair(File backupFile) {
        String backupPath = backupFile.getAbsolutePath();
        File sha256File = new File(backupPath + ".sha256");
        if (backupFile.delete()) {
            logger.info("Deleted old backup file: {}", (Object)backupFile.getName());
        } else {
            logger.error("Failed to delete backup file: {}", (Object)backupFile.getName());
        }
        if (sha256File.exists()) {
            if (sha256File.delete()) {
                logger.info("Deleted old backup checksum file: {}", (Object)sha256File.getName());
            } else {
                logger.error("Failed to delete backup checksum file: {}", (Object)sha256File.getName());
            }
        }
    }

    private String extractTimestamp(File file) {
        Matcher matcher = BACKUP_FULL_PATTERN.matcher(file.getName());
        if (matcher.matches()) {
            return matcher.group(1);
        }
        matcher = BACKUP_CONFIG_PATTERN.matcher(file.getName());
        if (matcher.matches()) {
            return matcher.group(1);
        }
        return "";
    }

    public void scheduleCustomTask(String taskId, String cronExpression, String taskName, Runnable task) {
        Runnable wrappedTask = () -> {
            logger.info("Executing custom task: {}", (Object)taskName);
            try {
                task.run();
                logger.info("Custom task '{}' completed successfully", (Object)taskName);
            }
            catch (Exception e) {
                logger.error("Error executing custom task '{}'", (Object)taskName, (Object)e);
            }
        };
        this.scheduleTask(taskId, cronExpression, wrappedTask);
    }

    private String buildCronExpression(String expr, String defaultCron) {
        if (expr == null || expr.isEmpty()) {
            logger.warn("DB Retention cleanup schedule not configured, using default cron: {}", (Object)defaultCron);
            return defaultCron;
        }
        String raw = expr.trim();
        String lower = raw.toLowerCase();
        if ("minutely".equals(lower)) {
            String cron = "0 0/1 * * * ?";
            logger.info("Using minutely schedule -> cron {}", (Object)cron);
            return cron;
        }
        if (lower.startsWith("hourly")) {
            int minute = 0;
            int colon = raw.indexOf(":");
            if (colon > 0 && colon < raw.length() - 1) {
                try {
                    minute = Integer.parseInt(raw.substring(colon + 1).trim());
                    if (minute < 0 || minute > 59) {
                        throw new NumberFormatException("Minute out of range");
                    }
                }
                catch (NumberFormatException ex) {
                    logger.error("Invalid hourly minute '{}', using 0", (Object)raw.substring(colon + 1).trim());
                    minute = 0;
                }
            }
            String cron = String.format("0 %d * * * ?", minute);
            logger.info("Using hourly schedule {} -> cron {}", (Object)raw, (Object)cron);
            return cron;
        }
        if (lower.startsWith("cron:")) {
            String cron = raw.substring(5).trim();
            if (cron.isEmpty()) {
                logger.error("Empty cron expression after 'cron:' prefix, using default {}", (Object)defaultCron);
                return defaultCron;
            }
            logger.info("Using explicit cron expression: {}", (Object)cron);
            return cron;
        }
        try {
            LocalTime time = LocalTime.parse(raw);
            String cron = String.format("0 %d %d * * ?", time.getMinute(), time.getHour());
            logger.info("Using daily schedule {} -> cron {}", (Object)raw, (Object)cron);
            return cron;
        }
        catch (DateTimeParseException ex) {
            logger.error("Invalid DB Retention cleanup schedule '{}', using default cron {}", (Object)raw, (Object)defaultCron);
            return defaultCron;
        }
    }

    @PreDestroy
    public void cleanup() {
        logger.info("Shutting down CronScheduler, cancelling {} tasks", (Object)this.scheduledTasks.size());
        this.scheduledTasks.forEach((taskId, task) -> {
            if (task != null && !task.isCancelled()) {
                task.cancel(false);
                logger.debug("Cancelled task: {}", taskId);
            }
        });
        this.scheduledTasks.clear();
        logger.info("CronScheduler shutdown complete");
    }
}

