/*
 * Decompiled with CFR 0.152.
 */
package com.coraltele.helper;

import com.coraltele.helper.Constants;
import com.coraltele.helper.RequestResponse;
import com.coraltele.model.DatabaseInfo;
import com.coraltele.service.user.service.UserService;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.ClassPathResource;
import org.springframework.jdbc.datasource.DriverManagerDataSource;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

@Service
public class BackupRestoreHelper {
    @Autowired
    UserService userService;
    private static final Logger logger = LogManager.getLogger(BackupRestoreHelper.class);
    private final ObjectMapper objectMapper;
    private static final String BACKUP_ROOT_PATH = "/mnt/backup";

    public BackupRestoreHelper(ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        File backupDir = new File(BACKUP_ROOT_PATH);
        if (!backupDir.exists()) {
            logger.info("Backup root directory does not exist. Creating: {}", (Object)BACKUP_ROOT_PATH);
            if (backupDir.mkdirs()) {
                logger.info("Successfully created backup root directory: {}", (Object)BACKUP_ROOT_PATH);
            } else {
                logger.error("Failed to create backup root directory: {}", (Object)BACKUP_ROOT_PATH);
            }
        } else {
            logger.debug("Backup root directory already exists: {}", (Object)BACKUP_ROOT_PATH);
        }
    }

    private List<DatabaseInfo> loadDatabaseConfig() throws IOException {
        logger.info("Loading database configuration from databases.json");
        ClassPathResource resource = new ClassPathResource("static-data/backup/databases.json");
        try (InputStream inputStream = resource.getInputStream();){
            List databases = (List)this.objectMapper.readValue(inputStream, (TypeReference)new /* Unavailable Anonymous Inner Class!! */);
            logger.info("Loaded {} database(s) from configuration", (Object)databases.size());
            List list = databases;
            return list;
        }
    }

    private List<String> loadAdditionalFilesConfig() throws IOException {
        logger.info("Loading additional files configuration from additonal-files.json");
        ClassPathResource resource = new ClassPathResource("static-data/backup/additonal-files.json");
        try (InputStream inputStream = resource.getInputStream();){
            List filePaths = (List)this.objectMapper.readValue(inputStream, (TypeReference)new /* Unavailable Anonymous Inner Class!! */);
            logger.info("Loaded {} additional file(s) from configuration", (Object)filePaths.size());
            List list = filePaths;
            return list;
        }
    }

    private long calculateDatabaseSize(DatabaseInfo dbInfo, boolean fullBackup) {
        logger.info("Calculating size for database: {} (fullBackup: {})", (Object)dbInfo.getDbName(), (Object)fullBackup);
        DriverManagerDataSource dataSource = new DriverManagerDataSource();
        dataSource.setDriverClassName("org.postgresql.Driver");
        dataSource.setUrl("jdbc:postgresql://127.0.0.1:5432/" + dbInfo.getDbName());
        dataSource.setUsername("postgres");
        dataSource.setPassword("");
        long totalSize = 0L;
        try (Connection conn = dataSource.getConnection();){
            StringBuilder query = new StringBuilder("SELECT COALESCE(SUM(pg_total_relation_size(quote_ident(schemaname) || '.' || quote_ident(tablename))), 0) AS total_size FROM pg_tables WHERE schemaname NOT IN ('pg_catalog', 'information_schema')");
            if (!fullBackup && dbInfo.getIgnoreTables() != null && !dbInfo.getIgnoreTables().isEmpty()) {
                query.append(" AND (schemaname || '.' || tablename) NOT IN (");
                for (int i = 0; i < dbInfo.getIgnoreTables().size(); ++i) {
                    query.append(i == 0 ? "?" : ", ?");
                }
                query.append(")");
            }
            try (PreparedStatement stmt = conn.prepareStatement(query.toString());){
                if (!fullBackup && dbInfo.getIgnoreTables() != null && !dbInfo.getIgnoreTables().isEmpty()) {
                    for (int i = 0; i < dbInfo.getIgnoreTables().size(); ++i) {
                        stmt.setString(i + 1, (String)dbInfo.getIgnoreTables().get(i));
                    }
                }
                try (ResultSet rs = stmt.executeQuery();){
                    if (rs.next()) {
                        totalSize = rs.getLong("total_size");
                        double sizeInMB = (double)totalSize / 1048576.0;
                        if (fullBackup) {
                            logger.info("Database '{}' FULL size: {:.2f} MB ({} bytes)", (Object)dbInfo.getDbName(), (Object)sizeInMB, (Object)totalSize);
                        } else {
                            logger.info("Database '{}' size (excluding ignored tables): {:.2f} MB ({} bytes)", (Object)dbInfo.getDbName(), (Object)sizeInMB, (Object)totalSize);
                            if (dbInfo.getIgnoreTables() != null && !dbInfo.getIgnoreTables().isEmpty()) {
                                logger.info("Excluded tables: {}", (Object)dbInfo.getIgnoreTables());
                            }
                        }
                    }
                }
            }
        }
        catch (SQLException e) {
            logger.error("Error calculating size for database: {}", (Object)dbInfo.getDbName(), (Object)e);
        }
        return totalSize;
    }

    private boolean checkDiskSpace(String backupPath, long requiredSize, double safetyMultiplier) {
        File backupDir = new File(backupPath);
        if (!backupDir.exists()) {
            logger.info("Creating backup directory: {}", (Object)backupPath);
            if (!backupDir.mkdirs()) {
                logger.error("Failed to create backup directory: {}", (Object)backupPath);
                return false;
            }
        }
        long freeSpace = backupDir.getFreeSpace();
        long requiredWithSafety = (long)((double)requiredSize * safetyMultiplier);
        double freeSpaceGB = (double)freeSpace / 1.073741824E9;
        double requiredGB = (double)requiredSize / 1.073741824E9;
        double requiredWithSafetyGB = (double)requiredWithSafety / 1.073741824E9;
        logger.info("Disk space check for: {}", (Object)backupPath);
        logger.info("Available space: {:.2f} GB", (Object)freeSpaceGB);
        logger.info("Required space (estimated): {:.2f} GB", (Object)requiredGB);
        logger.info("Required space (with {:.1f}x safety margin): {:.2f} GB", (Object)safetyMultiplier, (Object)requiredWithSafetyGB);
        if (freeSpace < requiredWithSafety) {
            logger.error("Insufficient disk space! Available: {:.2f} GB, Required: {:.2f} GB", (Object)freeSpaceGB, (Object)requiredWithSafetyGB);
            return false;
        }
        logger.info("Sufficient disk space available");
        return true;
    }

    private boolean dumpDatabase(DatabaseInfo dbInfo, String backupPath, String timestamp, boolean fullBackup) {
        String dbName = dbInfo.getDbName();
        String backupDir = backupPath + "/" + timestamp;
        String dumpFileName = String.format("%s/%s.db", backupDir, dbName);
        File backupDirectory = new File(backupDir);
        if (!backupDirectory.exists()) {
            logger.info("Creating backup directory: {}", (Object)backupDir);
            if (!backupDirectory.mkdirs()) {
                logger.error("Failed to create backup directory: {}", (Object)backupDir);
                return false;
            }
        }
        logger.info("Starting dump for database: {}", (Object)dbName);
        logger.info("Dump file: {}", (Object)dumpFileName);
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("pg_dump");
            command.add("-h");
            command.add("127.0.0.1");
            command.add("-p");
            command.add("5432");
            command.add("-U");
            command.add("postgres");
            command.add("-Fc");
            command.add("-f");
            command.add(dumpFileName);
            if (!fullBackup && dbInfo.getIgnoreTables() != null && !dbInfo.getIgnoreTables().isEmpty()) {
                logger.info("Excluding data from {} table(s)", (Object)dbInfo.getIgnoreTables().size());
                for (String tableName : dbInfo.getIgnoreTables()) {
                    command.add("--exclude-table-data=" + tableName);
                    logger.info("  - Excluding data from: {}", (Object)tableName);
                }
            } else if (fullBackup) {
                logger.info("Full backup mode - including all tables with data");
            }
            command.add(dbName);
            logger.debug("Executing command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            processBuilder.environment().put("PGPASSWORD", "");
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("pg_dump output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Successfully dumped database: {}", (Object)dbName);
                File dumpFile = new File(dumpFileName);
                if (!dumpFile.exists()) {
                    logger.error("Dump file not found: {}", (Object)dumpFileName);
                    return false;
                }
                long fileSize = dumpFile.length();
                if (fileSize == 0L) {
                    logger.error("Dump file is empty: {}", (Object)dumpFileName);
                    return false;
                }
                double fileSizeMB = (double)fileSize / 1048576.0;
                logger.info("Dump file size: {:.2f} MB ({} bytes)", (Object)fileSizeMB, (Object)fileSize);
                return true;
            }
            logger.error("pg_dump failed for database {} with exit code: {}", (Object)dbName, (Object)exitCode);
            if (output.length() > 0) {
                logger.error("pg_dump output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error executing pg_dump for database: {}", (Object)dbName, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean verifyDumpFile(String dumpFileName, String dbName) {
        logger.info("Verifying dump file integrity for database: {}", (Object)dbName);
        logger.info("Dump file: {}", (Object)dumpFileName);
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("pg_restore");
            command.add("--no-data");
            command.add("-f");
            command.add("/dev/null");
            command.add(dumpFileName);
            logger.debug("Executing verification command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("pg_restore output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Dump file verification successful for database: {}", (Object)dbName);
                return true;
            }
            logger.error("Dump file verification failed for database {} with exit code: {}", (Object)dbName, (Object)exitCode);
            if (output.length() > 0) {
                logger.error("pg_restore output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error verifying dump file for database: {}", (Object)dbName, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean generateHashFile(String dumpFileName, String dbName) {
        logger.info("Generating SHA-256 hash for database: {}", (Object)dbName);
        String hashFileName = dumpFileName + ".sha256";
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("bash");
            command.add("-c");
            command.add(String.format("cd $(dirname '%s') && sha256sum $(basename '%s') > $(basename '%s')", dumpFileName, dumpFileName, hashFileName));
            logger.debug("Executing hash command for: {}", (Object)dumpFileName);
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("sha256sum output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                File hashFile = new File(hashFileName);
                if (hashFile.exists() && hashFile.length() > 0L) {
                    logger.info("SHA-256 hash file created: {}", (Object)hashFileName);
                    return true;
                }
                logger.error("Hash file not created or empty: {}", (Object)hashFileName);
                return false;
            }
            logger.error("sha256sum failed for database {} with exit code: {}", (Object)dbName, (Object)exitCode);
            if (output.length() > 0) {
                logger.error("sha256sum output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error generating hash for database: {}", (Object)dbName, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean backupAdditionalFile(String filePath, String backupPath, String timestamp) {
        logger.info("Backing up additional file: {}", (Object)filePath);
        File sourceFile = new File(filePath);
        if (!sourceFile.exists()) {
            logger.error("Source file does not exist: {}", (Object)filePath);
            return false;
        }
        if (!sourceFile.isFile()) {
            logger.error("Source path is not a file: {}", (Object)filePath);
            return false;
        }
        String backupDir = backupPath + "/" + timestamp;
        String destinationPath = backupDir + "/additional-files/" + sourceFile.getName();
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("cp");
            command.add("-p");
            command.add(filePath);
            command.add(destinationPath);
            File destFile = new File(destinationPath);
            File destDir = destFile.getParentFile();
            if (!destDir.exists()) {
                logger.info("Creating directory for additional files: {}", (Object)destDir.getAbsolutePath());
                if (!destDir.mkdirs()) {
                    logger.error("Failed to create directory: {}", (Object)destDir.getAbsolutePath());
                    return false;
                }
            }
            logger.debug("Executing command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("cp output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                File copiedFile = new File(destinationPath);
                if (copiedFile.exists() && copiedFile.length() > 0L) {
                    logger.info("Successfully backed up file to: {}", (Object)destinationPath);
                    boolean hashGenerated = this.generateHashFile(destinationPath, sourceFile.getName());
                    if (!hashGenerated) {
                        logger.warn("Hash generation failed for additional file: {}", (Object)sourceFile.getName());
                    }
                    return true;
                }
                logger.error("Copied file not found or empty: {}", (Object)destinationPath);
                return false;
            }
            logger.error("cp command failed for file {} with exit code: {}", (Object)filePath, (Object)exitCode);
            if (output.length() > 0) {
                logger.error("cp output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error backing up file: {}", (Object)filePath, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private String compressBackupFolder(String backupPath, String timestamp, String backupType) {
        String backupDir = backupPath + "/" + timestamp;
        String archiveName = "backup_" + backupType + "_" + timestamp + ".tar.gz";
        String archivePath = backupPath + "/" + archiveName;
        logger.info("Starting compression of backup folder: {}", (Object)backupDir);
        logger.info("Archive will be created at: {}", (Object)archivePath);
        try {
            File folder = new File(backupDir);
            long folderSize = this.calculateFolderSize(folder);
            double folderSizeMB = (double)folderSize / 1048576.0;
            double folderSizeGB = (double)folderSize / 1.073741824E9;
            logger.info("Backup folder size: {:.2f} MB ({:.2f} GB)", (Object)folderSizeMB, (Object)folderSizeGB);
            long freeSpace = new File(backupPath).getFreeSpace();
            double freeSpaceGB = (double)freeSpace / 1.073741824E9;
            logger.info("Available disk space: {:.2f} GB", (Object)freeSpaceGB);
            if (freeSpace < folderSize) {
                logger.error("Insufficient disk space for compression! Available: {:.2f} GB, Required: ~{:.2f} GB", (Object)freeSpaceGB, (Object)folderSizeGB);
                return null;
            }
            ArrayList<String> command = new ArrayList<String>();
            command.add("tar");
            command.add("-czf");
            command.add(archivePath);
            command.add("-C");
            command.add(backupPath);
            command.add(timestamp);
            logger.debug("Executing compression command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("tar output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                File archiveFile = new File(archivePath);
                if (archiveFile.exists() && archiveFile.length() > 0L) {
                    long archiveSize = archiveFile.length();
                    double archiveSizeMB = (double)archiveSize / 1048576.0;
                    double archiveSizeGB = (double)archiveSize / 1.073741824E9;
                    double compressionRatio = (1.0 - (double)archiveSize / (double)folderSize) * 100.0;
                    logger.info("Archive created successfully: {}", (Object)archivePath);
                    logger.info("Archive size: {:.2f} MB ({:.2f} GB)", (Object)archiveSizeMB, (Object)archiveSizeGB);
                    logger.info("Compression ratio: {:.1f}%", (Object)compressionRatio);
                    return archivePath;
                }
                logger.error("Archive file not created or empty: {}", (Object)archivePath);
                return null;
            }
            logger.error("tar command failed with exit code: {}", (Object)exitCode);
            if (output.length() > 0) {
                logger.error("tar output: {}", (Object)output.toString());
            }
            return null;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error compressing backup folder: {}", (Object)backupDir, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return null;
        }
    }

    private long calculateFolderSize(File folder) {
        long size = 0L;
        if (folder.isFile()) {
            return folder.length();
        }
        File[] files = folder.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isFile()) {
                    size += file.length();
                    continue;
                }
                if (!file.isDirectory()) continue;
                size += this.calculateFolderSize(file);
            }
        }
        return size;
    }

    private boolean validateBackupArchive(String archivePath, String timestamp, List<String> expectedDatabases, List<String> expectedAdditionalFiles) {
        logger.info("Validating backup archive: {}", (Object)archivePath);
        File archiveFile = new File(archivePath);
        if (!archiveFile.exists()) {
            logger.error("Archive file does not exist: {}", (Object)archivePath);
            return false;
        }
        try {
            ArrayList<String> testCommand = new ArrayList<String>();
            testCommand.add("tar");
            testCommand.add("-tzf");
            testCommand.add(archivePath);
            logger.info("Testing archive integrity...");
            logger.debug("Executing command: {}", (Object)String.join((CharSequence)" ", testCommand));
            ProcessBuilder testBuilder = new ProcessBuilder(testCommand);
            testBuilder.redirectErrorStream(true);
            Process testProcess = testBuilder.start();
            ArrayList<String> archiveContents = new ArrayList<String>();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(testProcess.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    archiveContents.add(line);
                    output.append(line).append("\n");
                }
            }
            int exitCode = testProcess.waitFor();
            if (exitCode != 0) {
                logger.error("Archive integrity test failed with exit code: {}", (Object)exitCode);
                return false;
            }
            logger.info("Archive integrity test passed. Found {} entries in archive", (Object)archiveContents.size());
            logger.info("Archive contents (first 20 entries):");
            for (int i = 0; i < Math.min(20, archiveContents.size()); ++i) {
                logger.info("  {}", archiveContents.get(i));
            }
            if (archiveContents.size() > 20) {
                logger.info("  ... and {} more entries", (Object)(archiveContents.size() - 20));
            }
            logger.info("Verifying expected database files...");
            int foundDatabases = 0;
            int foundHashFiles = 0;
            for (String dbName : expectedDatabases) {
                String expectedDbFile = timestamp + "/" + dbName + ".db";
                String expectedHashFile = timestamp + "/" + dbName + ".db.sha256";
                boolean dbFileFound = archiveContents.stream().anyMatch(entry -> entry.equals(expectedDbFile));
                boolean hashFileFound = archiveContents.stream().anyMatch(entry -> entry.equals(expectedHashFile));
                if (dbFileFound) {
                    ++foundDatabases;
                    logger.debug("Found database dump: {}", (Object)expectedDbFile);
                } else {
                    logger.warn("Missing database dump: {}", (Object)expectedDbFile);
                }
                if (hashFileFound) {
                    ++foundHashFiles;
                    logger.debug("Found hash file: {}", (Object)expectedHashFile);
                    continue;
                }
                logger.warn("Missing hash file: {}", (Object)expectedHashFile);
            }
            logger.info("Database validation: {}/{} database dumps found, {}/{} hash files found", (Object)foundDatabases, (Object)expectedDatabases.size(), (Object)foundHashFiles, (Object)expectedDatabases.size());
            int foundAdditionalFiles = 0;
            int foundAdditionalHashFiles = 0;
            if (expectedAdditionalFiles != null && !expectedAdditionalFiles.isEmpty()) {
                logger.info("Verifying expected additional files...");
                for (String filePath : expectedAdditionalFiles) {
                    File file = new File(filePath);
                    String fileName = file.getName();
                    String expectedFile = timestamp + "/additional-files/" + fileName;
                    boolean fileFound = archiveContents.stream().anyMatch(entry -> entry.equals(expectedFile));
                    if (fileFound) {
                        ++foundAdditionalFiles;
                        logger.debug("Found additional file: {}", (Object)expectedFile);
                        continue;
                    }
                    logger.warn("Missing additional file: {}", (Object)expectedFile);
                }
                logger.info("Additional files validation: {}/{} files found", (Object)foundAdditionalFiles, (Object)expectedAdditionalFiles.size());
            }
            boolean validationPassed = true;
            if (foundDatabases < expectedDatabases.size()) {
                logger.error("Archive validation failed: Missing database dump files");
                validationPassed = false;
            }
            if (foundHashFiles < expectedDatabases.size()) {
                logger.warn("Archive validation warning: Missing database hash files (non-critical)");
            }
            if (expectedAdditionalFiles != null && !expectedAdditionalFiles.isEmpty()) {
                if (foundAdditionalFiles < expectedAdditionalFiles.size()) {
                    logger.error("Archive validation failed: Missing additional files");
                    validationPassed = false;
                }
                if (foundAdditionalHashFiles > 0) {
                    logger.info("Additional file hash files found: {}/{}", (Object)foundAdditionalHashFiles, (Object)expectedAdditionalFiles.size());
                }
            }
            if (validationPassed) {
                logger.info("Archive validation successful - all expected files present");
            } else {
                logger.error("Archive validation failed - some files are missing");
            }
            return validationPassed;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error validating archive: {}", (Object)archivePath, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean deleteFolder(File folder) {
        File[] files;
        if (!folder.exists()) {
            return true;
        }
        if (folder.isDirectory() && (files = folder.listFiles()) != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    if (this.deleteFolder(file)) continue;
                    return false;
                }
                if (file.delete()) continue;
                logger.warn("Failed to delete file: {}", (Object)file.getAbsolutePath());
                return false;
            }
        }
        return folder.delete();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private int getPostgresVersion() {
        try {
            DriverManagerDataSource dataSource = new DriverManagerDataSource();
            dataSource.setDriverClassName("org.postgresql.Driver");
            dataSource.setUrl("jdbc:postgresql://127.0.0.1:5432/postgres");
            dataSource.setUsername("postgres");
            dataSource.setPassword("");
            try (Connection conn = dataSource.getConnection();
                 PreparedStatement stmt = conn.prepareStatement("SHOW server_version");
                 ResultSet rs = stmt.executeQuery();){
                if (!rs.next()) return -1;
                String version = rs.getString(1);
                String majorVersion = version.split("\\.")[0];
                int versionNumber = Integer.parseInt(majorVersion);
                logger.info("Detected PostgreSQL version: {}", (Object)versionNumber);
                int n = versionNumber;
                return n;
            }
        }
        catch (NumberFormatException | SQLException e) {
            logger.error("Error detecting PostgreSQL version", (Throwable)e);
        }
        return -1;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean dropDatabase(String dbName, int postgresVersion) {
        logger.info("Dropping database: {} (PostgreSQL {})", (Object)dbName, (Object)postgresVersion);
        try {
            DriverManagerDataSource dataSource = new DriverManagerDataSource();
            dataSource.setDriverClassName("org.postgresql.Driver");
            dataSource.setUrl("jdbc:postgresql://127.0.0.1:5432/postgres");
            dataSource.setUsername("postgres");
            dataSource.setPassword("");
            Connection conn = dataSource.getConnection();
            Throwable throwable2 = null;
            conn.setAutoCommit(true);
            if (postgresVersion >= 13) {
                logger.info("Using DROP DATABASE WITH (FORCE) for PostgreSQL {}", (Object)postgresVersion);
                String dropSql = String.format("DROP DATABASE IF EXISTS %s WITH (FORCE)", dbName);
                try (PreparedStatement stmt2 = conn.prepareStatement(dropSql);){
                    stmt2.execute();
                    logger.info("Successfully dropped database: {}", (Object)dbName);
                    boolean bl = true;
                    return bl;
                }
            }
            logger.info("Using connection termination method for PostgreSQL {}", (Object)postgresVersion);
            String disallowConnSql = String.format("UPDATE pg_database SET datallowconn = 'false' WHERE datname = '%s'", dbName);
            try (PreparedStatement stmt3 = conn.prepareStatement(disallowConnSql);){
                stmt3.execute();
                logger.debug("Disallowed new connections to database: {}", (Object)dbName);
            }
            String limitSql = String.format("ALTER DATABASE %s CONNECTION LIMIT 0", dbName);
            try (PreparedStatement stmt4 = conn.prepareStatement(limitSql);){
                stmt4.execute();
                logger.debug("Set connection limit to 0 for database: {}", (Object)dbName);
            }
            String terminateSql = String.format("SELECT pg_terminate_backend(pid) FROM pg_stat_activity WHERE datname = '%s'", dbName);
            try (PreparedStatement stmt5 = conn.prepareStatement(terminateSql);){
                stmt5.execute();
                logger.debug("Terminated existing connections to database: {}", (Object)dbName);
            }
            Thread.sleep(1000L);
            String dropSql = String.format("DROP DATABASE IF EXISTS %s", dbName);
            PreparedStatement stmt = conn.prepareStatement(dropSql);
            Throwable throwable = null;
            try {
                stmt.execute();
                logger.info("Successfully dropped database: {}", (Object)dbName);
                boolean bl = true;
                return bl;
            }
            catch (Throwable throwable4) {
                try {
                    throwable = throwable4;
                    throw throwable4;
                }
                catch (Throwable throwable3) {
                    throwable2 = throwable3;
                    throw throwable3;
                }
                finally {
                    if (stmt != null) {
                        if (throwable != null) {
                            try {
                                stmt.close();
                            }
                            catch (Throwable throwable5) {
                                throwable.addSuppressed(throwable5);
                            }
                        } else {
                            stmt.close();
                        }
                    }
                }
            }
            finally {
                if (conn != null) {
                    if (throwable2 != null) {
                        try {
                            conn.close();
                        }
                        catch (Throwable throwable6) {
                            throwable2.addSuppressed(throwable6);
                        }
                    } else {
                        conn.close();
                    }
                }
            }
        }
        catch (InterruptedException | SQLException e) {
            logger.error("Error dropping database: {}", (Object)dbName, (Object)e);
            if (!(e instanceof InterruptedException)) return false;
            Thread.currentThread().interrupt();
            return false;
        }
    }

    /*
     * Exception decompiling
     */
    private boolean createDatabase(String dbName) {
        /*
         * This method has failed to decompile.  When submitting a bug report, please provide this stack trace, and (if you hold appropriate legal rights) the relevant class file.
         * 
         * org.benf.cfr.reader.util.ConfusedCFRException: Started 2 blocks at once
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.getStartingBlocks(Op04StructuredStatement.java:412)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op04StructuredStatement.buildNestedBlocks(Op04StructuredStatement.java:487)
         *     at org.benf.cfr.reader.bytecode.analysis.opgraph.Op03SimpleStatement.createInitialStructuredBlock(Op03SimpleStatement.java:736)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisInner(CodeAnalyser.java:850)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysisOrWrapFail(CodeAnalyser.java:278)
         *     at org.benf.cfr.reader.bytecode.CodeAnalyser.getAnalysis(CodeAnalyser.java:201)
         *     at org.benf.cfr.reader.entities.attributes.AttributeCode.analyse(AttributeCode.java:94)
         *     at org.benf.cfr.reader.entities.Method.analyse(Method.java:531)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseMid(ClassFile.java:1055)
         *     at org.benf.cfr.reader.entities.ClassFile.analyseTop(ClassFile.java:942)
         *     at org.benf.cfr.reader.Driver.doJarVersionTypes(Driver.java:257)
         *     at org.benf.cfr.reader.Driver.doJar(Driver.java:139)
         *     at org.benf.cfr.reader.CfrDriverImpl.analyse(CfrDriverImpl.java:76)
         *     at org.benf.cfr.reader.Main.main(Main.java:54)
         */
        throw new IllegalStateException("Decompilation failed");
    }

    private boolean restoreDatabase(String dbName, String dumpFilePath) {
        logger.info("Restoring database: {} from {}", (Object)dbName, (Object)dumpFilePath);
        File dumpFile = new File(dumpFilePath);
        if (!dumpFile.exists()) {
            logger.error("Dump file not found: {}", (Object)dumpFilePath);
            return false;
        }
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("pg_restore");
            command.add("-h");
            command.add("127.0.0.1");
            command.add("-p");
            command.add("5432");
            command.add("-U");
            command.add("postgres");
            command.add("-d");
            command.add(dbName);
            command.add("-v");
            command.add(dumpFilePath);
            logger.debug("Executing restore command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            processBuilder.environment().put("PGPASSWORD", "");
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("pg_restore output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Successfully restored database: {}", (Object)dbName);
                return true;
            }
            logger.error("pg_restore failed for database {} with exit code: {}", (Object)dbName, (Object)exitCode);
            if (output.length() > 0) {
                logger.error("pg_restore output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error executing pg_restore for database: {}", (Object)dbName, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean restoreCoralFile(String backupFilePath) {
        String destinationPath;
        logger.info("Restoring coral configuration file from: {}", (Object)backupFilePath);
        File sourceFile = new File(backupFilePath);
        if (!sourceFile.exists()) {
            logger.error("Backup coral file not found: {}", (Object)backupFilePath);
            return false;
        }
        File wipFile = new File("/etc/default/coral.wip");
        if (wipFile.exists()) {
            destinationPath = "/etc/default/coral.wip";
            logger.info("coral.wip exists, copying to: {}", (Object)destinationPath);
        } else {
            destinationPath = "/etc/default/coral";
            logger.info("coral.wip does not exist, copying to: {}", (Object)destinationPath);
        }
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("sudo");
            command.add("cp");
            command.add("-p");
            command.add(backupFilePath);
            command.add(destinationPath);
            logger.debug("Executing command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("cp output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Successfully restored coral file to: {}", (Object)destinationPath);
                return true;
            }
            logger.error("Failed to copy coral file with exit code: {}", (Object)exitCode);
            if (output.length() > 0) {
                logger.error("cp output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error restoring coral file", (Throwable)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean triggerConfigureNode() {
        logger.info("Triggering configure node...");
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("/usr/bin/configure-node");
            logger.debug("Executing command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.info("configure-node output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Successfully triggered configure node");
                this.userService.regenerateSIPCredentials();
                logger.info("Regenerate SIP credentials completed");
                logger.info("Scheduling system reboot in 10 seconds...");
                Thread rebootThread = new Thread(() -> {
                    try {
                        logger.warn("Reboot thread started - system will reboot in 10 seconds...");
                        logger.warn("All active connections will be terminated.");
                        Thread.sleep(10000L);
                        ArrayList<String> rebootCommand = new ArrayList<String>();
                        rebootCommand.add("sudo");
                        rebootCommand.add("shutdown");
                        rebootCommand.add("-r");
                        rebootCommand.add("now");
                        ProcessBuilder rebootBuilder = new ProcessBuilder(rebootCommand);
                        rebootBuilder.redirectErrorStream(true);
                        Process rebootProcess = rebootBuilder.start();
                        StringBuilder rebootOutput = new StringBuilder();
                        try (BufferedReader reader = new BufferedReader(new InputStreamReader(rebootProcess.getInputStream()));){
                            String line;
                            while ((line = reader.readLine()) != null) {
                                rebootOutput.append(line).append("\n");
                                logger.info("shutdown output: {}", (Object)line);
                            }
                        }
                        int rebootExitCode = rebootProcess.waitFor();
                        if (rebootExitCode == 0) {
                            logger.info("System reboot scheduled successfully");
                        } else {
                            logger.error("Failed to schedule reboot, exit code: {}", (Object)rebootExitCode);
                            if (rebootOutput.length() > 0) {
                                logger.error("shutdown command output: {}", (Object)rebootOutput.toString());
                            }
                        }
                    }
                    catch (IOException e) {
                        logger.error("Failed to trigger system reboot", (Throwable)e);
                        logger.warn("Manual reboot required after restore");
                    }
                    catch (InterruptedException e) {
                        logger.error("Reboot thread was interrupted", (Throwable)e);
                        Thread.currentThread().interrupt();
                    }
                }, "RebootThread");
                rebootThread.setDaemon(true);
                rebootThread.start();
                logger.info("Reboot thread started - API will return immediately");
                return true;
            }
            logger.error("configure-node failed with exit code: {}", (Object)exitCode);
            if (output.length() > 0) {
                logger.error("configure-node output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error triggering configure node", (Throwable)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean verifyFileHash(String filePath, String hashFilePath) {
        logger.info("Verifying hash for file: {}", (Object)filePath);
        File file = new File(filePath);
        File hashFile = new File(hashFilePath);
        if (!file.exists()) {
            logger.error("File not found: {}", (Object)filePath);
            return false;
        }
        if (!hashFile.exists()) {
            logger.error("Hash file not found: {}", (Object)hashFilePath);
            return false;
        }
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("bash");
            command.add("-c");
            command.add(String.format("cd $(dirname '%s') && sha256sum -c $(basename '%s')", filePath, hashFilePath));
            logger.debug("Executing hash verification command");
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("sha256sum -c output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Hash verification successful for: {}", (Object)filePath);
                return true;
            }
            logger.error("Hash verification failed for: {}", (Object)filePath);
            if (output.length() > 0) {
                logger.error("sha256sum output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error verifying hash for file: {}", (Object)filePath, (Object)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    private boolean extractBackupArchive(String archivePath, String extractPath) {
        logger.info("Extracting backup archive: {} to {}", (Object)archivePath, (Object)extractPath);
        File archiveFile = new File(archivePath);
        if (!archiveFile.exists()) {
            logger.error("Archive file not found: {}", (Object)archivePath);
            return false;
        }
        File extractDir = new File(extractPath);
        if (!extractDir.exists() && !extractDir.mkdirs()) {
            logger.error("Failed to create extraction directory: {}", (Object)extractPath);
            return false;
        }
        try {
            ArrayList<String> command = new ArrayList<String>();
            command.add("tar");
            command.add("-xzf");
            command.add(archivePath);
            command.add("-C");
            command.add(extractPath);
            logger.debug("Executing extraction command: {}", (Object)String.join((CharSequence)" ", command));
            ProcessBuilder processBuilder = new ProcessBuilder(command);
            processBuilder.redirectErrorStream(true);
            Process process = processBuilder.start();
            StringBuilder output = new StringBuilder();
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));){
                String line;
                while ((line = reader.readLine()) != null) {
                    output.append(line).append("\n");
                    logger.debug("tar output: {}", (Object)line);
                }
            }
            int exitCode = process.waitFor();
            if (exitCode == 0) {
                logger.info("Successfully extracted archive");
                return true;
            }
            logger.error("Archive extraction failed with exit code: {}", (Object)exitCode);
            if (output.length() > 0) {
                logger.error("tar output: {}", (Object)output.toString());
            }
            return false;
        }
        catch (IOException | InterruptedException e) {
            logger.error("Error extracting archive", (Throwable)e);
            if (e instanceof InterruptedException) {
                Thread.currentThread().interrupt();
            }
            return false;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public RequestResponse restoreDatabases(String archive) {
        String archivePath;
        RequestResponse response = new RequestResponse();
        if ((archive = archive.trim()).endsWith(".tar.gz")) {
            archivePath = "/mnt/backup/" + archive;
        } else if (archive.startsWith("full_") || archive.startsWith("config_")) {
            archivePath = "/mnt/backup/backup_" + archive + ".tar.gz";
        } else {
            if (!archive.matches("\\d{8}_\\d{6}")) {
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Invalid archive format: " + archive);
                return response;
            }
            File fullFile = new File("/mnt/backup/backup_full_" + archive + ".tar.gz");
            File configFile = new File("/mnt/backup/backup_config_" + archive + ".tar.gz");
            File oldFile = new File("/mnt/backup/backup_" + archive + ".tar.gz");
            if (fullFile.exists()) {
                archivePath = fullFile.getAbsolutePath();
            } else if (configFile.exists()) {
                archivePath = configFile.getAbsolutePath();
            } else {
                if (!oldFile.exists()) {
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Backup archive not found for timestamp: " + archive);
                    return response;
                }
                archivePath = oldFile.getAbsolutePath();
            }
        }
        logger.info("=== Starting Database Restore Process ===");
        logger.info("Archive: {}", (Object)archivePath);
        try {
            String dumpFilePath;
            String dbName;
            File archiveFile = new File(archivePath);
            if (!archiveFile.exists()) {
                logger.error("Archive file not found: {}", (Object)archivePath);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Archive file not found: " + archivePath);
                return response;
            }
            int postgresVersion = this.getPostgresVersion();
            if (postgresVersion == -1) {
                logger.error("Unable to determine PostgreSQL version. Aborting restore.");
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Unable to determine PostgreSQL version. Database may not be running.");
                return response;
            }
            String extractPath = "/tmp/restore_" + System.currentTimeMillis();
            logger.info("Creating temporary extraction directory: {}", (Object)extractPath);
            if (!this.extractBackupArchive(archivePath, extractPath)) {
                logger.error("Failed to extract backup archive");
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Failed to extract backup archive. File may be corrupted.");
                return response;
            }
            File extractDir = new File(extractPath);
            File[] subdirs = extractDir.listFiles(File::isDirectory);
            if (subdirs == null || subdirs.length == 0) {
                logger.error("No timestamp directory found in extracted archive");
                this.deleteFolder(extractDir);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Invalid backup archive structure. No timestamp directory found.");
                return response;
            }
            File timestampDir = subdirs[0];
            String timestamp = timestampDir.getName();
            logger.info("Found backup timestamp directory: {}", (Object)timestamp);
            List databases = this.loadDatabaseConfig();
            logger.info("Found {} database(s) to restore", (Object)databases.size());
            int successCount = 0;
            int failCount = 0;
            logger.info("=== Pre-restore Hash Verification Phase ===");
            logger.info("Verifying integrity of all dump files before restore...");
            for (DatabaseInfo dbInfo : databases) {
                dbName = dbInfo.getDbName();
                dumpFilePath = timestampDir.getAbsolutePath() + "/" + dbName + ".db";
                String hashFilePath = dumpFilePath + ".sha256";
                File dumpFile = new File(dumpFilePath);
                if (!dumpFile.exists()) {
                    logger.error("Dump file not found for database {}: {}", (Object)dbName, (Object)dumpFilePath);
                    logger.error("ABORTING RESTORE - Missing dump file");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Dump file not found for database: " + dbName + ". Archive is incomplete.");
                    return response;
                }
                File hashFile = new File(hashFilePath);
                if (!hashFile.exists()) {
                    logger.error("Hash file not found for database {}: {}", (Object)dbName, (Object)hashFilePath);
                    logger.error("ABORTING RESTORE - Cannot proceed without hash verification. SECURITY RISK!");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Hash file not found for database: " + dbName + ". Cannot verify integrity - SECURITY RISK!");
                    return response;
                }
                logger.info("Verifying hash for database: {}", (Object)dbName);
                if (!this.verifyFileHash(dumpFilePath, hashFilePath)) {
                    logger.error("Hash verification FAILED for database: {}", (Object)dbName);
                    logger.error("ABORTING RESTORE - Dump file may be corrupted or tampered with!");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Hash verification FAILED for database: " + dbName + ". File may be corrupted or tampered with!");
                    return response;
                }
                logger.info("Hash verification PASSED for database: {}", (Object)dbName);
            }
            logger.info("=== All hash verifications PASSED ===");
            logger.info("Proceeding with database restore...");
            for (DatabaseInfo dbInfo : databases) {
                dbName = dbInfo.getDbName();
                dumpFilePath = timestampDir.getAbsolutePath() + "/" + dbName + ".db";
                logger.info("=== Restoring database: {} ===", (Object)dbName);
                if (!this.dropDatabase(dbName, postgresVersion)) {
                    logger.error("Failed to drop database: {}", (Object)dbName);
                    logger.error("ABORTING RESTORE - Cannot continue if database drop fails");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Failed to drop existing database: " + dbName + ". Active connections may be preventing drop.");
                    return response;
                }
                if (!this.createDatabase(dbName)) {
                    logger.error("Failed to create database: {}", (Object)dbName);
                    logger.error("ABORTING RESTORE - Cannot continue if database creation fails");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Failed to create database: " + dbName + ". Check PostgreSQL permissions.");
                    return response;
                }
                if (!this.restoreDatabase(dbName, dumpFilePath)) {
                    logger.error("Failed to restore database: {}", (Object)dbName);
                    logger.error("ABORTING RESTORE - Database restore failed");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Failed to restore database: " + dbName + " from dump file. Check pg_restore logs.");
                    return response;
                }
                ++successCount;
                logger.info("Successfully restored database: {}", (Object)dbName);
            }
            logger.info("Database restore completed. Success: {}, Failed: {}", (Object)successCount, (Object)failCount);
            String archiveHashFile = archivePath + ".sha256";
            File archiveHash = new File(archiveHashFile);
            if (archiveHash.exists()) {
                logger.info("Verifying integrity of backup archive...");
                if (!this.verifyFileHash(archivePath, archiveHashFile)) {
                    logger.error("Archive hash verification FAILED!");
                    logger.error("ABORTING - Archive may be corrupted or tampered with.");
                    this.deleteFolder(extractDir);
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Archive hash verification FAILED! Archive may be corrupted or tampered with.");
                    return response;
                }
                logger.info("Archive hash verification PASSED");
            } else {
                logger.warn("Archive hash file not found: {}", (Object)archiveHashFile);
                logger.warn("Continuing without verification!");
            }
            File additionalFilesDir = new File(timestampDir, "additional-files");
            if (additionalFilesDir.exists() && additionalFilesDir.isDirectory()) {
                logger.info("Restoring additional files...");
                File coralFile = new File(additionalFilesDir, "coral");
                if (coralFile.exists()) {
                    File coralHashFile = new File(additionalFilesDir, "coral.sha256");
                    if (!coralHashFile.exists()) {
                        logger.error("Coral hash file not found");
                        logger.error("ABORTING - Cannot proceed without coral file hash verification!");
                        this.deleteFolder(extractDir);
                        response.setStatus(Constants.Error.ERROR);
                        response.setMessage("Coral hash file not found. Cannot verify integrity - SECURITY RISK!");
                        return response;
                    }
                    logger.info("Verifying integrity of coral file...");
                    if (!this.verifyFileHash(coralFile.getAbsolutePath(), coralHashFile.getAbsolutePath())) {
                        logger.error("Coral file hash verification FAILED!");
                        logger.error("ABORTING - Coral file may be corrupted.");
                        this.deleteFolder(extractDir);
                        response.setStatus(Constants.Error.ERROR);
                        response.setMessage("Coral file hash verification FAILED! File may be corrupted.");
                        return response;
                    }
                    logger.info("Coral file hash verification PASSED");
                    if (!this.restoreCoralFile(coralFile.getAbsolutePath())) {
                        logger.error("Failed to restore coral configuration file");
                        logger.error("ABORTING - Coral file restoration failed");
                        this.deleteFolder(extractDir);
                        response.setStatus(Constants.Error.ERROR);
                        response.setMessage("Failed to restore coral configuration file. Check sudo permissions.");
                        return response;
                    }
                } else {
                    logger.info("No coral file found in backup - skipping");
                }
            } else {
                logger.info("No additional files directory found in backup - skipping");
            }
            logger.info("Triggering configure node...");
            if (!this.triggerConfigureNode()) {
                logger.error("Failed to trigger configure node");
                logger.error("ABORTING - Manual configuration required");
                this.deleteFolder(extractDir);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Failed to trigger configure-node. Manual configuration may be required.");
                return response;
            }
            logger.info("Cleaning up temporary extraction directory...");
            this.deleteFolder(extractDir);
            logger.info("=== Database Restore Completed Successfully ===");
            logger.info("All {} database(s) restored and verified", (Object)successCount);
            logger.warn("SYSTEM WILL REBOOT IN 10 SECONDS!");
            response.setStatus(Constants.Error.OK);
            response.setMessage("Database restore completed successfully. " + successCount + " database(s) restored and verified. SYSTEM REBOOTING IN 10 SECONDS!");
            return response;
        }
        catch (IOException e) {
            logger.error("Error during database restore", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Error during database restore: " + e.getMessage());
            return response;
        }
        catch (Exception e) {
            logger.error("Unexpected error during database restore", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Unexpected error during restore: " + e.getMessage());
            return response;
        }
    }

    public RequestResponse listBackups() {
        RequestResponse response = new RequestResponse();
        try {
            File backupDir = new File(BACKUP_ROOT_PATH);
            if (!backupDir.exists() || !backupDir.isDirectory()) {
                logger.warn("Backup directory does not exist: {}", (Object)BACKUP_ROOT_PATH);
                response.setStatus(Constants.Error.OK);
                response.setMessage("No backups found. Backup directory does not exist.");
                response.setData(new ArrayList());
                return response;
            }
            File[] files = backupDir.listFiles((dir, name) -> name.startsWith("backup_") && name.endsWith(".tar.gz"));
            if (files == null || files.length == 0) {
                logger.info("No backup files found in: {}", (Object)BACKUP_ROOT_PATH);
                response.setStatus(Constants.Error.OK);
                response.setMessage("No backups found.");
                response.setData(new ArrayList());
                return response;
            }
            ArrayList backupList = new ArrayList();
            for (File file : files) {
                String timestamp;
                LinkedHashMap<String, Object> backupInfo = new LinkedHashMap<String, Object>();
                String fileName = file.getName();
                String backupType = "config";
                if (fileName.matches("backup_(full|config)_\\d{8}_\\d{6}\\.tar\\.gz")) {
                    String[] parts = fileName.replace("backup_", "").replace(".tar.gz", "").split("_", 2);
                    backupType = parts[0];
                    timestamp = parts[1];
                } else {
                    timestamp = fileName.replace("backup_", "").replace(".tar.gz", "");
                }
                backupInfo.put("filename", fileName);
                backupInfo.put("timestamp", timestamp);
                backupInfo.put("backupType", backupType);
                backupInfo.put("path", file.getAbsolutePath());
                backupInfo.put("size", file.length());
                backupInfo.put("sizeMB", String.format("%.2f", (double)file.length() / 1048576.0));
                backupInfo.put("sizeGB", String.format("%.2f", (double)file.length() / 1.073741824E9));
                backupInfo.put("lastModified", file.lastModified());
                File hashFile = new File(file.getAbsolutePath() + ".sha256");
                backupInfo.put("hasHash", hashFile.exists());
                backupList.add(backupInfo);
            }
            backupList.sort((a, b) -> ((String)b.get("timestamp")).compareTo((String)a.get("timestamp")));
            logger.info("Found {} backup(s)", (Object)backupList.size());
            response.setStatus(Constants.Error.OK);
            response.setMessage("Found " + backupList.size() + " backup(s)");
            response.setData(backupList);
            return response;
        }
        catch (Exception e) {
            logger.error("Error listing backups", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Error listing backups: " + e.getMessage());
            return response;
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public RequestResponse deleteBackup(String timestamp) {
        RequestResponse response = new RequestResponse();
        try {
            String archiveFileName;
            if (timestamp == null || timestamp.trim().isEmpty()) {
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Timestamp parameter is required");
                return response;
            }
            if ((timestamp = timestamp.trim()).startsWith("full_") || timestamp.startsWith("config_")) {
                archiveFileName = "backup_" + timestamp + ".tar.gz";
            } else {
                if (!timestamp.matches("\\d{8}_\\d{6}")) {
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Invalid timestamp format: " + timestamp);
                    return response;
                }
                File fullFile = new File("/mnt/backup/backup_full_" + timestamp + ".tar.gz");
                File configFile = new File("/mnt/backup/backup_config_" + timestamp + ".tar.gz");
                File oldFile = new File("/mnt/backup/backup_" + timestamp + ".tar.gz");
                if (fullFile.exists()) {
                    archiveFileName = "backup_full_" + timestamp + ".tar.gz";
                } else if (configFile.exists()) {
                    archiveFileName = "backup_config_" + timestamp + ".tar.gz";
                } else {
                    if (!oldFile.exists()) {
                        response.setStatus(Constants.Error.ERROR);
                        response.setMessage("Backup not found for timestamp: " + timestamp);
                        return response;
                    }
                    archiveFileName = "backup_" + timestamp + ".tar.gz";
                }
            }
            String archivePath = "/mnt/backup/" + archiveFileName;
            String hashPath = archivePath + ".sha256";
            File archiveFile = new File(archivePath);
            File hashFile = new File(hashPath);
            if (!archiveFile.exists()) {
                logger.error("Backup archive not found: {}", (Object)archivePath);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Backup archive not found: " + archiveFileName);
                return response;
            }
            long archiveSize = archiveFile.length();
            double archiveSizeMB = (double)archiveSize / 1048576.0;
            boolean archiveDeleted = archiveFile.delete();
            if (!archiveDeleted) {
                logger.error("Failed to delete backup archive: {}", (Object)archivePath);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Failed to delete backup archive: " + archiveFileName);
                return response;
            }
            logger.info("Deleted backup archive: {} ({:.2f} MB)", (Object)archiveFileName, (Object)archiveSizeMB);
            boolean hashDeleted = false;
            if (hashFile.exists()) {
                hashDeleted = hashFile.delete();
                if (hashDeleted) {
                    logger.info("Deleted hash file: {}", (Object)hashFile.getName());
                } else {
                    logger.warn("Failed to delete hash file: {}", (Object)hashFile.getName());
                }
            }
            response.setStatus(Constants.Error.OK);
            response.setMessage(String.format("Successfully deleted backup: %s (%.2f MB)", archiveFileName, archiveSizeMB));
            LinkedHashMap<String, Object> deletionInfo = new LinkedHashMap<String, Object>();
            deletionInfo.put("filename", archiveFileName);
            deletionInfo.put("timestamp", timestamp);
            deletionInfo.put("sizeMB", String.format("%.2f", archiveSizeMB));
            deletionInfo.put("hashDeleted", hashDeleted);
            response.setData(deletionInfo);
            return response;
        }
        catch (Exception e) {
            logger.error("Error deleting backup", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Error deleting backup: " + e.getMessage());
            return response;
        }
    }

    public RequestResponse backupDatabases(boolean fullBackup) {
        RequestResponse response = new RequestResponse();
        try {
            logger.info("=== Starting Database Backup Process ===");
            logger.info("Backup mode: {}", (Object)(fullBackup ? "FULL (all tables with data)" : "SELECTIVE (excluding configured tables)"));
            List databases = this.loadDatabaseConfig();
            long totalBackupSize = 0L;
            for (DatabaseInfo dbInfo : databases) {
                logger.info("Processing database: {}", (Object)dbInfo.getDbName());
                long dbSize = this.calculateDatabaseSize(dbInfo, fullBackup);
                totalBackupSize += dbSize;
            }
            double totalSizeInMB = (double)totalBackupSize / 1048576.0;
            double totalSizeInGB = (double)totalBackupSize / 1.073741824E9;
            logger.info("Total estimated backup size: {:.2f} MB ({:.2f} GB)", (Object)totalSizeInMB, (Object)totalSizeInGB);
            String backupPath = BACKUP_ROOT_PATH;
            if (!this.checkDiskSpace(backupPath, totalBackupSize, 2.5)) {
                logger.error("Aborting backup due to insufficient disk space");
                response.setStatus(Constants.Error.ERROR);
                response.setMessage(String.format("Insufficient disk space. Required: %.2f GB (with safety margin)", totalSizeInGB * 2.5));
                return response;
            }
            String timestamp = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss"));
            String backupType = fullBackup ? "full" : "config";
            logger.info("Starting database dumps with timestamp: {} (type: {})", (Object)timestamp, (Object)backupType);
            int successCount = 0;
            int failCount = 0;
            ArrayList<String> dumpedFiles = new ArrayList<String>();
            for (Object dbInfo : databases) {
                boolean success = this.dumpDatabase((DatabaseInfo)dbInfo, backupPath, timestamp, fullBackup);
                if (success) {
                    String dumpFileName = String.format("%s/%s/%s.db", backupPath, timestamp, dbInfo.getDbName());
                    boolean verified = this.verifyDumpFile(dumpFileName, dbInfo.getDbName());
                    if (verified) {
                        boolean hashGenerated = this.generateHashFile(dumpFileName, dbInfo.getDbName());
                        if (hashGenerated) {
                            ++successCount;
                            dumpedFiles.add(dumpFileName);
                            continue;
                        }
                        ++failCount;
                        logger.error("Hash generation failed for database: {}", (Object)dbInfo.getDbName());
                        continue;
                    }
                    ++failCount;
                    logger.error("Verification failed for database: {}", (Object)dbInfo.getDbName());
                    continue;
                }
                ++failCount;
            }
            List additionalFiles = new ArrayList();
            try {
                additionalFiles = this.loadAdditionalFilesConfig();
                for (String filePath : additionalFiles) {
                    boolean success = this.backupAdditionalFile(filePath, backupPath, timestamp);
                    if (success) {
                        logger.info("Successfully backed up additional file: {}", (Object)filePath);
                        this.generateHashFile(filePath, new File(filePath).getName());
                        continue;
                    }
                    logger.warn("Failed to backup additional file: {}", (Object)filePath);
                }
            }
            catch (IOException e) {
                logger.error("Failed to load additional files configuration", (Throwable)e);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Failed to load additional files configuration: " + e.getMessage());
                return response;
            }
            logger.info("Database dump, verification, and hash generation completed. Success: {}, Failed: {}", (Object)successCount, (Object)failCount);
            if (failCount > 0) {
                logger.warn("Backup completed with {} failed database(s), but proceeding with compression", (Object)failCount);
            }
            logger.info("Starting backup compression...");
            String archivePath = this.compressBackupFolder(backupPath, timestamp, backupType);
            if (archivePath != null) {
                logger.info("Backup compression successful");
                logger.info("Generating hash for backup archive...");
                boolean archiveHashGenerated = this.generateHashFile(archivePath, "backup_archive");
                if (!archiveHashGenerated) {
                    logger.error("Archive hash generation failed");
                    response.setStatus(Constants.Error.ERROR);
                    response.setMessage("Backup compressed but hash generation failed. Backup may not be secure.");
                    return response;
                }
                logger.info("Archive hash generation successful");
                logger.info("Validating backup archive...");
                ArrayList<String> dbNames = new ArrayList<String>();
                for (DatabaseInfo dbInfo : databases) {
                    dbNames.add(dbInfo.getDbName());
                }
                boolean validationPassed = this.validateBackupArchive(archivePath, timestamp, dbNames, additionalFiles);
                if (validationPassed) {
                    logger.info("Archive validation successful");
                    String uncompressedFolder = backupPath + "/" + timestamp;
                    logger.info("Deleting uncompressed backup folder: {}", (Object)uncompressedFolder);
                    if (this.deleteFolder(new File(uncompressedFolder))) {
                        logger.info("Successfully deleted uncompressed folder");
                    } else {
                        logger.warn("Failed to delete uncompressed folder - manual cleanup may be needed");
                    }
                    response.setStatus(Constants.Error.OK);
                    response.setMessage(String.format("Backup completed successfully. %d database(s) backed up. Archive: %s", successCount, new File(archivePath).getName()));
                    return response;
                }
                logger.error("Archive validation failed - backup may be incomplete");
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Archive validation failed. Backup may be incomplete or corrupted.");
                return response;
            }
            logger.error("Backup compression failed");
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Backup compression failed. Check disk space and permissions.");
            return response;
        }
        catch (IOException e) {
            logger.error("Failed to load database configuration", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Failed to load database configuration: " + e.getMessage());
            return response;
        }
        catch (Exception e) {
            logger.error("Unexpected error during backup", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Unexpected error during backup: " + e.getMessage());
            return response;
        }
    }

    public File getBackupFile(String timestamp) {
        File backupDir = new File(BACKUP_ROOT_PATH);
        if (!backupDir.exists() || !backupDir.isDirectory()) {
            logger.error("Backup directory does not exist: {}", (Object)BACKUP_ROOT_PATH);
            return null;
        }
        if (!(timestamp = timestamp.trim()).startsWith("full_") && !timestamp.startsWith("config_")) {
            if (timestamp.matches("\\d{8}_\\d{6}")) {
                File fullFile = new File("/mnt/backup/backup_full_" + timestamp + ".tar.gz");
                File configFile = new File("/mnt/backup/backup_config_" + timestamp + ".tar.gz");
                File oldFile = new File("/mnt/backup/backup_" + timestamp + ".tar.gz");
                if (fullFile.exists()) {
                    return fullFile;
                }
                if (configFile.exists()) {
                    return configFile;
                }
                if (oldFile.exists()) {
                    return oldFile;
                }
                logger.error("Backup file not found for timestamp: {}", (Object)timestamp);
                return null;
            }
            logger.error("Invalid timestamp format: {}", (Object)timestamp);
            return null;
        }
        String archiveFileName = "backup_" + timestamp + ".tar.gz";
        File archiveFile = new File("/mnt/backup/" + archiveFileName);
        if (archiveFile.exists()) {
            logger.info("Found backup file: {}", (Object)archiveFile.getAbsolutePath());
            return archiveFile;
        }
        logger.error("Backup file not found: {}", (Object)archiveFile.getAbsolutePath());
        return null;
    }

    public RequestResponse uploadBackupFile(MultipartFile file) {
        RequestResponse response = new RequestResponse();
        try {
            if (file == null || file.isEmpty()) {
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("No file uploaded");
                return response;
            }
            String originalFilename = file.getOriginalFilename();
            if (originalFilename == null || !originalFilename.startsWith("backup_") || !originalFilename.endsWith(".tar.gz")) {
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Invalid backup file name. Must start with 'backup_' and end with '.tar.gz'");
                return response;
            }
            File backupDir = new File(BACKUP_ROOT_PATH);
            if (!backupDir.exists() && !backupDir.mkdirs()) {
                logger.error("Failed to create backup directory: {}", (Object)BACKUP_ROOT_PATH);
                response.setStatus(Constants.Error.ERROR);
                response.setMessage("Failed to create backup directory");
                return response;
            }
            File destFile = new File(BACKUP_ROOT_PATH, originalFilename);
            try (InputStream is = file.getInputStream();){
                Files.copy(is, destFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            long fileSize = destFile.length();
            double fileSizeMB = (double)fileSize / 1048576.0;
            logger.info("Uploaded backup file: {} ({:.2f} MB)", (Object)destFile.getAbsolutePath(), (Object)fileSizeMB);
            response.setStatus(Constants.Error.OK);
            response.setMessage(String.format("Backup file uploaded successfully: %s (%.2f MB)", originalFilename, fileSizeMB));
            LinkedHashMap<String, Object> fileInfo = new LinkedHashMap<String, Object>();
            fileInfo.put("filename", originalFilename);
            fileInfo.put("path", destFile.getAbsolutePath());
            fileInfo.put("size", fileSize);
            fileInfo.put("sizeMB", String.format("%.2f", fileSizeMB));
            response.setData(fileInfo);
            return response;
        }
        catch (IOException e) {
            logger.error("Error uploading backup file", (Throwable)e);
            response.setStatus(Constants.Error.ERROR);
            response.setMessage("Error uploading backup file: " + e.getMessage());
            return response;
        }
    }
}

